use crate::constants::ApiOptions;
use crate::constants::CookieNames;
use crate::types::ParsedResponse;
use crate::utils::cookie::get_cookie_value;
use fastly::Request;
use std::collections::HashMap;

type QueryParams = HashMap<String, String>;


pub fn fetch_new_key(request: &Request, server_url: &str, project_key: &str, segment_key: &str) -> Option<ParsedResponse> {
	let mut query_params = QueryParams::new();
	query_params.insert(ApiOptions::OPCODE.to_string(), ApiOptions::REQUEST.to_string());
	query_params.insert(ApiOptions::SERVICE_ID.to_string(), project_key.to_string());
	query_params.insert(ApiOptions::ACTION_ID.to_string(), segment_key.to_string());

	if let Some(user_id) = get_cookie_value(request, CookieNames::USER_ID) {
		query_params.insert("user_data".to_string(), user_id);
	}

	let url = build_api_url(server_url, &query_params);
	println!("API Request URL: {}", url);

	let api_request = Request::get(&url);
	match api_request.send("nf_core") {
		Ok(response) => {
			let body_str = response.into_body_str();
			println!("API Response: {}", body_str);
			parse_response(&body_str)
		}
		Err(e) => {
			println!("Error in fetch_new_key: {:?}", e);
			None
		}
	}
}

pub fn fetch_renew_key(request: &Request, server_url: &str, key: &str, sticky: Option<&str>) -> Option<ParsedResponse> {
	let mut query_params = QueryParams::new();
	query_params.insert(ApiOptions::OPCODE.to_string(), ApiOptions::RENEW.to_string());
	query_params.insert("key".to_string(), key.to_string());

	if let Some(sticky_val) = sticky {
		query_params.insert("sticky".to_string(), sticky_val.to_string());
	}

	if let Some(user_id) = get_cookie_value(request, CookieNames::USER_ID) {
		query_params.insert("user_data".to_string(), user_id);
	}

	let url = build_api_url(server_url, &query_params);
	println!("API Request URL: {}", url);

	let api_request = Request::get(&url);
	match api_request.send("nf_core") {
		Ok(response) => {
			let body_str = response.into_body_str();
			println!("API Response: {}", body_str);
			parse_response(&body_str)
		}
		Err(e) => {
			println!("Error in fetch_renew_key: {:?}", e);
			None
		}
	}
}

fn build_api_url(server_url: &str, query_params: &QueryParams) -> String {
	let mut url = format!("{}/{}", server_url, ApiOptions::PATH);

	if !query_params.is_empty() {
		let query_string: Vec<String> = query_params.iter().map(|(k, v)| format!("{}={}", urlencoding::encode(k), urlencoding::encode(v))).collect();
		url.push_str("?");
		url.push_str(&query_string.join("&"));
	}

	url
}

fn parse_response(response_text: &str) -> Option<ParsedResponse> {
	let response_text = response_text.trim();
	
	let parts: Vec<&str> = response_text.splitn(2, ':').collect();
	if parts.len() != 2 {
		println!("Invalid response format - no ':' separator found: {}", response_text);
		return None;
	}
	
	let code = parts[0].trim();
	let rest_data = parts[1].trim();
	
	if code.is_empty() {
		println!("Invalid response format - empty code: {}", response_text);
		return None;
	}
	
	let mut extra = HashMap::new();
	
	if !rest_data.is_empty() {
		for pair in rest_data.split('&') {
			if let Some((key, value)) = pair.split_once('=') {
				let key = key.trim();
				let value = value.trim();
				if !key.is_empty() && !value.is_empty() {
					extra.insert(key.to_string(), value.to_string());
				}
			}
		}
	}
	
	Some(ParsedResponse { 
		code: code.to_string(), 
		extra 
	})
}
