use crate::types::NFConfig;
use fastly::config_store::ConfigStore;
use once_cell::sync::OnceCell;
use std::sync::Mutex;

static CONFIG: OnceCell<Mutex<Option<NFConfig>>> = OnceCell::new();

pub fn handle_config() -> bool {
	let service_version = std::env::var("FASTLY_SERVICE_VERSION").unwrap_or_default();

	let config = if service_version == "0" || service_version.is_empty() {
		NFConfig {
			client_id: "leonar-1486".to_string(),
			server_url: "https://leonar-1486.netfunnel.staging.stclab.com".to_string(),
			setting_url: "https://nf-setting-bucket.staging.stclab.com/leonar-1486.netfunnel/nf-setting.json".to_string(),
			vwr_page_url: "https://agent-lib.staging.stclab.com/vwr-page/leonar-1486.netfunnel/index.html".to_string(),
			good_bots: vec!["Firefox".to_string()],
			return_key: false,
			cookie_domain: "".to_string(),
		}
	} else {
		let config_store = ConfigStore::open("nf_config_store");
		let client_id = config_store.get("CLIENT_ID").unwrap_or_default();
		let mut server_url = String::new();
		let mut setting_url = String::new();
		let mut vwr_page_url = String::new();

		if !client_id.is_empty() {
			server_url = format!("https://{}.netfunnel.stclab.com", client_id);
			setting_url = format!("https://nf-setting-bucket.stclab.com/{}.netfunnel/nf-setting.json", client_id);
			vwr_page_url = format!("https://agent-lib.stclab.com/vwr-page/{}.netfunnel/index.html", client_id);
		}

		if let Some(s) = config_store.get("SERVER_URL") {
			if !s.is_empty() {
				server_url = s;
			}
		}

		if let Some(s) = config_store.get("SETTING_URL") {
			if !s.is_empty() {
				setting_url = s;
			}
		}

		if let Some(s) = config_store.get("VWR_PAGE_URL") {
			if !s.is_empty() {
				vwr_page_url = s;
			}
		}

		if let Some(domain) = config_store.get("VWR_PAGE_DOMAIN") {
			if !domain.is_empty() && !client_id.is_empty() {
				vwr_page_url = format!("{}/vwr-page/{}.netfunnel/index.html", domain, client_id);
			}
		}

		if server_url.is_empty() || setting_url.is_empty() || vwr_page_url.is_empty() {
		    return false;
		}

		let good_bots = config_store.get("GOOD_BOTS")
			.unwrap_or_default()
			.split(',')
			.map(|s| s.trim().to_string())
			.filter(|s| !s.is_empty())
			.collect();

		let return_key = config_store.get("RETURN_KEY")
			.unwrap_or_default()
			.parse()
			.unwrap_or(false);

		let cookie_domain = config_store.get("COOKIE_DOMAIN")
			.unwrap_or_default();

		NFConfig {
			client_id,
			server_url,
			setting_url,
			vwr_page_url,
			good_bots,
			return_key,
			cookie_domain,
		}
	};

	CONFIG.set(Mutex::new(Some(config))).map_err(|_| println!("Failed to set config")).is_ok()
}

pub fn get_config() -> Option<NFConfig> {
	CONFIG.get()?.lock().ok()?.clone()
}

