use crate::api::{fetch_new_key, fetch_renew_key};
use crate::config::get_config;
use crate::constants::{CookieNames, QueryNames};
use crate::types::{InvalidateKey, MatchedKeys, ParsedResponse, SettingJson};
use crate::utils::{cookie, is_good_bot, rule_match, url_utils};
use fastly::http::{header, StatusCode};
use fastly::{Error, Request, Response};
use url::Url;

pub fn handle_good_bots(request: &Request) -> bool {
	let config = match get_config() {
		Some(config) => config,
		None => return false,
	};

	let user_agent = match request.get_header("user-agent") {
		Some(ua) => ua.to_str().unwrap_or(""),
		None => return false,
	};

	is_good_bot(user_agent, &config.good_bots)
}

pub fn handle_query_string(_request: &Request, url: &Url) -> Option<Response> {
	let query_params = url_utils::get_query_params(url);
	
	let nf_pjk = query_params.get(QueryNames::NF_PJK)?;
	let nf_smk = query_params.get(QueryNames::NF_SMK)?;
	let nf_key = query_params.get(QueryNames::NF_KEY);
	let nf_stk = query_params.get(QueryNames::NF_STK);
	
	if nf_pjk.is_empty() && nf_smk.is_empty() {
		return None;
	}
	
	let mut clean_url = url.clone();
	url_utils::remove_nf_queries(&mut clean_url);
	
	let cookie_name = cookie::generate_cookie_name(nf_pjk, nf_smk);
	
	let mut response = Response::from_status(StatusCode::FOUND)
		.with_header(header::LOCATION, clean_url.as_str())
		.with_header(header::CACHE_CONTROL, "no-store, no-cache, must-revalidate, max-age=0, private")
		.with_header(header::PRAGMA, "no-cache")
		.with_header(header::EXPIRES, "0");
	
	if let Some(key) = nf_key {
		let cookie_value = cookie::generate_cookie_value(key, nf_stk.map_or("", |v| v));
		let key_cookie = cookie::generate_cookie(&cookie_name, false, Some(&cookie_value));
		let redirected_cookie = cookie::generate_cookie(CookieNames::REDIRECTED, false, Some("true"));
		
		response.append_header(header::SET_COOKIE, &redirected_cookie);
		response.append_header(header::SET_COOKIE, &key_cookie);
	} else {
		let key_cookie = cookie::generate_cookie(&cookie_name, true, None);
		response.append_header(header::SET_COOKIE, &key_cookie);
	}
	
	Some(response)
}

pub fn handle_redirected(request: &Request) -> bool {
	cookie::get_cookie_value(request, CookieNames::REDIRECTED).map(|value| value == "true").unwrap_or(false)
}

pub fn handle_setting_json() -> Option<SettingJson> {
	let config = get_config()?;
	let setting_url = config.setting_url;

	match fetch_setting_json_sync(&setting_url) {
		Ok(setting) => {
			Some(setting)
		}
		Err(e) => {
			println!("Failed to load setting JSON: {:?}", e);
			None
		}
	}
}

pub fn handle_invalidate_key(projects: &[crate::types::Project], url: &str) -> Vec<InvalidateKey> {
	let mut invalidate_keys = Vec::new();
	
	for project in projects {
		for segment in &project.segments {
			if let Some(invalidate_url) = &segment.invalidate_key_url {
				if !invalidate_url.is_empty() && url.contains(invalidate_url) {
					invalidate_keys.push(InvalidateKey {
						invalidate_project_key: project.project_key.clone(),
						invalidate_segment_key: segment.segment_key.clone(),
					});
				}
			}
		}
	}
	
	invalidate_keys
}

pub fn handle_invalidate_keys_check(projects: &[crate::types::Project], url: &str) -> bool {
	let invalidate_keys = handle_invalidate_key(projects, url);
	!invalidate_keys.is_empty()
}

pub fn handle_invalidate_keys_response(request: Request, projects: &[crate::types::Project], url: &str) -> Result<Response, Error> {
	let invalidate_keys = handle_invalidate_key(projects, url);

	for invalidate_key in &invalidate_keys {
		println!("Invalidate key matched - project: {}, segment: {}", invalidate_key.invalidate_project_key, invalidate_key.invalidate_segment_key);
	}
	
	match request.send("origin") {
		Ok(mut origin_response) => {
			for invalidate_key in &invalidate_keys {
				let cookie_name = crate::utils::cookie::generate_cookie_name(
					&invalidate_key.invalidate_project_key, 
					&invalidate_key.invalidate_segment_key
				);
				let delete_cookie = crate::utils::cookie::generate_cookie(&cookie_name, true, None);
				origin_response.append_header(header::SET_COOKIE, &delete_cookie);
			}
			
			// 강력한 캐싱 방지 헤더 추가
			origin_response.set_header(header::CACHE_CONTROL, "no-store, no-cache, must-revalidate, max-age=0, private");
			origin_response.set_header(header::PRAGMA, "no-cache");
			origin_response.set_header(header::EXPIRES, "0");
			origin_response.remove_header(header::ETAG);
			origin_response.remove_header(header::LAST_MODIFIED);
			Ok(origin_response)
		}
		Err(e) => Err(e.into())
	}
}

pub fn handle_rule_match(projects: &[crate::types::Project], url: &str) -> Option<MatchedKeys> {
	rule_match::find_matched_rules(projects, url)
}

pub fn handle_cookie_existing(request: &Request, project_key: &str, segment_key: &str) -> Option<String> {
	let cookie_name = cookie::generate_cookie_name(project_key, segment_key);
	cookie::get_cookie_value(request, &cookie_name)
}

pub fn handle_renew_key(request: Request, url: &str, project_key: &str, segment_key: &str, key: &str, sticky: &str) -> Result<Response, Error> {
	let (request, config) = match get_config_or_send_origin(request) {
		Ok(result) => result,
		Err(response) => return Ok(response),
	};

	let sticky_param = if sticky.is_empty() { None } else { Some(sticky) };

	match fetch_renew_key(&request, &config.server_url, key, sticky_param) {
		Some(api_response) => {
			handle_response(request, url, project_key, segment_key, &api_response)
		}
		None => {
			println!("Failed to get response from fetch_renew_key");
			Ok(request.send("origin")?)
		}
	}
}

pub fn handle_new_key(request: Request, url: &str, project_key: &str, segment_key: &str) -> Result<Response, Error> {
	let (request, config) = match get_config_or_send_origin(request) {
		Ok(result) => result,
		Err(response) => return Ok(response),
	};

	match fetch_new_key(&request, &config.server_url, project_key, segment_key) {
		Some(api_response) => {
			handle_response(request, url, project_key, segment_key, &api_response)
		}
		None => {
			println!("Failed to get response from fetch_new_key");
			Ok(request.send("origin")?)
		}
	}
}

pub fn handle_response(request: Request, url: &str, project_key: &str, segment_key: &str, api_response: &ParsedResponse) -> Result<Response, Error> {
	let (request, config) = match get_config_or_send_origin(request) {
		Ok(result) => result,
		Err(response) => return Ok(response),
	};

	let cookie_name = cookie::generate_cookie_name(project_key, segment_key);
	let vwr_url = build_vwr_url(&request, &config.vwr_page_url, url, project_key, segment_key, api_response.extra.get("key"), api_response.extra.get("sticky"));

	match api_response.code.as_str() {
		"200" => {
			if let Some(key) = api_response.extra.get("key") {
				let empty_string = String::new();
				let sticky = api_response.extra.get("sticky").unwrap_or(&empty_string);
				let cookie_value = cookie::generate_cookie_value(key, sticky);
				send_origin_with_cookie(request, &cookie_name, false, Some(&cookie_value))
			} else {
				Ok(request.send("origin")?)
			}
		}
		"300" | "303" => {
			send_origin_with_cookie(request, &cookie_name, true, None)
		}
		"201" | "202" => {
			let is_wait = api_response.extra.get("vwr_type").map_or(false, |vwr_type| vwr_type == "wait");
			
			if is_wait && api_response.extra.get("key").is_some() {
				let key = api_response.extra.get("key").unwrap();
				let empty_string = String::new();
				let sticky = api_response.extra.get("sticky").unwrap_or(&empty_string);
				let cookie_value = cookie::generate_cookie_value(key, sticky);
				redirect_to_vwr_with_cookie(&vwr_url, &cookie_name, false, Some(&cookie_value))
			} else {
				redirect_to_vwr_with_cookie(&vwr_url, &cookie_name, true, None)
			}
		}
		"301" | "302" => {
			redirect_to_vwr_with_cookie(&vwr_url, &cookie_name, true, None)
		}
		"500" | "501" | "505" | "506" => {
			handle_new_key(request, url, project_key, segment_key)
		}
		_ => {
			Ok(request.send("origin")?)
		}
	}
}

fn get_config_or_send_origin(request: Request) -> Result<(Request, crate::types::NFConfig), Response> {
	match get_config() {
		Some(config) => Ok((request, config)),
		None => Err(request.send("origin").unwrap()),
	}
}

fn send_origin_with_cookie(request: Request, cookie_name: &str, is_remove: bool, cookie_value: Option<&str>) -> Result<Response, Error> {
	let mut origin_response = request.send("origin")?;
	let cookie_str = cookie::generate_cookie(cookie_name, is_remove, cookie_value);
	origin_response.set_header(header::SET_COOKIE, &cookie_str);
	
	// HTML response에 대해 강력한 캐싱 방지 헤더 추가
	origin_response.set_header(header::CACHE_CONTROL, "no-store, no-cache, must-revalidate, max-age=0, private");
	origin_response.set_header(header::PRAGMA, "no-cache");
	origin_response.set_header(header::EXPIRES, "0");
	
	// ETag와 Last-Modified 헤더 제거하여 브라우저 캐싱 완전 방지
	origin_response.remove_header(header::ETAG);
	origin_response.remove_header(header::LAST_MODIFIED);
	
	Ok(origin_response)
}

fn redirect_to_vwr_with_cookie(vwr_url: &str, cookie_name: &str, is_remove: bool, cookie_value: Option<&str>) -> Result<Response, Error> {
	let mut redirect_response = Response::from_status(StatusCode::FOUND)
		.with_header(header::LOCATION, vwr_url)
		.with_header(header::CACHE_CONTROL, "no-store, no-cache, must-revalidate, max-age=0, private")
		.with_header(header::PRAGMA, "no-cache")
		.with_header(header::EXPIRES, "0");
	
	let cookie_str = cookie::generate_cookie(cookie_name, is_remove, cookie_value);
	redirect_response.set_header(header::SET_COOKIE, &cookie_str);
	
	Ok(redirect_response)
}

fn build_vwr_url(request: &Request, vwr_page_url: &str, original_url: &str, project_key: &str, segment_key: &str, key: Option<&String>, sticky: Option<&String>) -> String {
	use crate::constants::QueryNames;
	
	let mut params = Vec::new();
	
	// Required parameters
	params.push(format!("{}={}", QueryNames::NF_PJK, urlencoding::encode(project_key)));
	params.push(format!("{}={}", QueryNames::NF_SMK, urlencoding::encode(segment_key)));
	params.push(format!("{}={}", QueryNames::NF_RDU, urlencoding::encode(original_url)));
	
	// Optional parameters
	if let Some(key_val) = key {
		params.push(format!("{}={}", QueryNames::NF_KEY, urlencoding::encode(key_val)));
	}
	
	if let Some(sticky_val) = sticky {
		params.push(format!("{}={}", QueryNames::NF_STK, urlencoding::encode(sticky_val)));
	}
	
	if let Some(user_id) = cookie::get_cookie_value(request, CookieNames::USER_ID) {
		params.push(format!("{}={}", QueryNames::NF_UDT, urlencoding::encode(&user_id)));
	}
	
	// Add return key flag from config
	let config = get_config();
	if let Some(cfg) = config {
		if !cfg.return_key {
			params.push(format!("{}=false", QueryNames::NF_RTK));
		}
	}
	
	format!("{}?{}", vwr_page_url, params.join("&"))
}

fn fetch_setting_json_sync(url: &str) -> Result<SettingJson, Error> {
	use std::time::{SystemTime, UNIX_EPOCH};
	
	let timestamp = SystemTime::now()
		.duration_since(UNIX_EPOCH)
		.unwrap()
		.as_millis();
	
	let cache_buster_url = format!("{}?ts={}", url, timestamp);
	
	let mut request = Request::get(&cache_buster_url);
	request.set_header("Cache-Control", "no-cache, no-store, must-revalidate, private");
	request.set_header("Pragma", "no-cache");
	request.set_header("Expires", "0");
	let response = request.send("nf_setting")?;
	let json_text = response.into_body_str();

	serde_json::from_str(&json_text).map_err(|e| {
		println!("Failed to parse setting JSON: {:?}", e);
		println!("Response was: {}", json_text);
		Error::msg("JSON parse error")
	})
}