use fastly::http::{header, Method};
use fastly::{Error, Request, Response};
use url::Url;

mod api;
mod config;
mod constants;
mod handlers;
mod types;
mod utils;

use config::handle_config;
use constants::CookieNames;
use handlers::*;

#[fastly::main]
fn main(request: Request) -> Result<Response, Error> {

	let request_url = match Url::parse(request.get_url_str()) {
		Ok(url) => {
			println!("Request URL: {}", url);
			url
		},
		Err(_) => return Ok(request.send("origin")?),
	};

	// method가 options || head면 에이전트 종료
	if matches!(request.get_method(), &Method::HEAD | &Method::OPTIONS) {
		return Ok(request.send("origin")?);
	}

	// request의 accept에 text/html이 포함되어있지 않으면 에이전트 종료
	if let Some(accept) = request.get_header("accept") {
		if let Ok(accept_str) = accept.to_str() {
			if !accept_str.contains("text/html") && !accept_str.contains("*/*") {
				return Ok(request.send("origin")?);
			}
		}
	}

	// 설정값 처리
	if !handle_config() {
		return Ok(request.send("origin")?);
	}

	// 검색 엔진 크롤러면 에이전트 종료
	if handle_good_bots(&request) {
		return Ok(request.send("origin")?);
	}

	// nf_ query가 있을 경우 제거하고 redirect
	if let Some(redirect_response) = handle_query_string(&request, &request_url) {
		return Ok(redirect_response);
	}

	// nf_ query를 제거하여 redirect된 경우 종료
	if handle_redirected(&request) {
		let mut origin_response = request.send("origin")?;
		let redirected_cookie = utils::cookie::generate_cookie(CookieNames::REDIRECTED, true, None);
		origin_response.set_header(header::SET_COOKIE, &redirected_cookie);
		
		// 강력한 캐싱 방지 헤더 추가
		origin_response.set_header(header::CACHE_CONTROL, "no-store, no-cache, must-revalidate, max-age=0, private");
		origin_response.set_header(header::PRAGMA, "no-cache");
		origin_response.set_header(header::EXPIRES, "0");
		origin_response.remove_header(header::ETAG);
		origin_response.remove_header(header::LAST_MODIFIED);
		
		return Ok(origin_response);
	}

	// setting.json 가져오기
	let setting_json = match crate::handlers::handle_setting_json() {
		Some(setting) => setting,
		None => return Ok(request.send("origin")?),
	};

	// requestURL에 invalidateKeyUrl이 포함되면 쿠키 제거 후 종료
	if let Some(projects) = &setting_json.data {
		if handle_invalidate_keys_check(projects, &request_url.to_string()) {
			return handle_invalidate_keys_response(request, projects, &request_url.to_string());
		}
	}

	// 트리거룰 매치
	let matched_keys = match setting_json.data.as_ref().and_then(|projects| handle_rule_match(projects, &request_url.to_string())) {
		Some(keys) => {
			println!("Trigger rule matched - project: {}, segment: {}", keys.matched_project_key, keys.matched_segment_key);
			keys
		}
		None => {
			return Ok(request.send("origin")?);
		}
	};

	// cookie에 넷퍼넬 키 존재하면 5002, 없으면 5101
	match handle_cookie_existing(&request, &matched_keys.matched_project_key, &matched_keys.matched_segment_key) {
		Some(cookie_value) => {
			let parts: Vec<&str> = cookie_value.split('&').collect();
			if parts.len() >= 1 && !parts[0].is_empty() {
				let key = parts[0];
				let sticky = parts.get(1).unwrap_or(&"");
				handle_renew_key(request, &request_url.to_string(), &matched_keys.matched_project_key, &matched_keys.matched_segment_key, key, sticky)
			} else {
				handle_new_key(request, &request_url.to_string(), &matched_keys.matched_project_key, &matched_keys.matched_segment_key)
			}
		}
		None => {
			handle_new_key(request, &request_url.to_string(), &matched_keys.matched_project_key, &matched_keys.matched_segment_key)
		}
	}
}
