use serde::{Deserialize, Deserializer, Serialize};

fn deserialize_number_as_u32<'de, D>(deserializer: D) -> Result<Option<u32>, D::Error>
where
	D: Deserializer<'de>,
{
	use serde::de::Error;

	let value: serde_json::Value = Deserialize::deserialize(deserializer)?;

	match value {
		serde_json::Value::Number(n) => {
			if let Some(i) = n.as_u64() {
				Ok(Some(i as u32))
			} else if let Some(f) = n.as_f64() {
				Ok(Some(f as u32))
			} else {
				Err(D::Error::custom("Invalid number format"))
			}
		}
		serde_json::Value::Null => Ok(None),
		_ => Err(D::Error::custom("Expected number or null")),
	}
}

fn deserialize_required_number_as_u32<'de, D>(deserializer: D) -> Result<u32, D::Error>
where
	D: Deserializer<'de>,
{
	use serde::de::Error;

	let value: serde_json::Value = Deserialize::deserialize(deserializer)?;

	match value {
		serde_json::Value::Number(n) => {
			if let Some(i) = n.as_u64() {
				Ok(i as u32)
			} else if let Some(f) = n.as_f64() {
				Ok(f as u32)
			} else {
				Err(D::Error::custom("Invalid number format"))
			}
		}
		_ => Err(D::Error::custom("Expected number")),
	}
}

#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct NFConfig {
	pub client_id: String,
	pub server_url: String,
	pub setting_url: String,
	pub vwr_page_url: String,
	pub good_bots: Vec<String>,
	pub return_key: bool,
	pub cookie_domain: String,
}

#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct ParsedResponse {
	pub code: String,
	#[serde(flatten)]
	pub extra: std::collections::HashMap<String, String>,
}

#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct SettingJson {
	#[serde(rename = "statusCode")]
	pub status_code: Option<u16>,
	pub status: Option<String>,
	#[serde(rename = "productType")]
	pub product_type: Option<String>,
	#[serde(rename = "eumEnable")]
	pub eum_enable: Option<bool>,
	#[serde(rename = "errorUrl")]
	pub error_url: Option<String>,
	#[serde(rename = "clientId")]
	pub client_id: Option<String>,
	pub data: Option<Vec<Project>>,
	pub error: Option<ErrorInfo>,
}

#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct ErrorInfo {
	#[serde(rename = "htmlUrl")]
	pub html_url: Option<String>,
}

#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct Project {
	#[serde(deserialize_with = "deserialize_required_number_as_u32")]
	pub id: u32,
	#[serde(rename = "projectName")]
	pub project_name: String,
	#[serde(rename = "projectKey")]
	pub project_key: String,
	#[serde(rename = "customerServiceDomain")]
	pub customer_service_domain: Option<String>,
	#[serde(rename = "activatedYn")]
	pub activated_yn: bool,
	#[serde(rename = "accessYn")]
	pub access_yn: bool,
	#[serde(rename = "segmentCount")]
	#[serde(deserialize_with = "deserialize_required_number_as_u32")]
	pub segment_count: u32,
	#[serde(rename = "sectionSegmentCount")]
	#[serde(deserialize_with = "deserialize_number_as_u32")]
	#[serde(default)]
	pub section_segment_count: Option<u32>,
	#[serde(rename = "blockYn")]
	pub block_yn: bool,
	pub segments: Vec<Segment>,
	#[serde(rename = "sectionSegments")]
	pub section_segments: Vec<SectionSegment>,
}

#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct Segment {
	#[serde(deserialize_with = "deserialize_number_as_u32")]
	#[serde(default)]
	pub id: Option<u32>,
	#[serde(rename = "vwrHost")]
	pub vwr_host: Option<String>,
	#[serde(rename = "vwrEntryRequestFlag")]
	pub vwr_entry_request_flag: Option<bool>,
	#[serde(rename = "segmentName")]
	pub segment_name: Option<String>,
	#[serde(rename = "segmentKey")]
	pub segment_key: String,
	#[serde(rename = "accessYn")]
	pub access_yn: Option<bool>,
	#[serde(rename = "segmentType")]
	pub segment_type: Option<String>,
	#[serde(rename = "maxInflow")]
	#[serde(deserialize_with = "deserialize_number_as_u32")]
	#[serde(default)]
	pub max_inflow: Option<u32>,
	#[serde(rename = "dynamicMinInflow")]
	#[serde(deserialize_with = "deserialize_number_as_u32")]
	#[serde(default)]
	pub dynamic_min_inflow: Option<u32>,
	#[serde(rename = "dynamicMaxInflow")]
	#[serde(deserialize_with = "deserialize_number_as_u32")]
	#[serde(default)]
	pub dynamic_max_inflow: Option<u32>,
	#[serde(rename = "dynamicSectionMinSec")]
	#[serde(deserialize_with = "deserialize_number_as_u32")]
	#[serde(default)]
	pub dynamic_section_min_sec: Option<u32>,
	#[serde(rename = "dynamicSectionMaxSec")]
	#[serde(deserialize_with = "deserialize_number_as_u32")]
	#[serde(default)]
	pub dynamic_section_max_sec: Option<u32>,
	pub uris: Option<Vec<Uri>>,
	#[serde(rename = "blockingPage")]
	pub blocking_page: Option<BlockingPage>,
	pub vwr: Option<Vwr>,
	#[serde(rename = "vwrPage")]
	pub vwr_page: Option<VwrPage>,
	#[serde(rename = "passedUserTimeout")]
	#[serde(deserialize_with = "deserialize_number_as_u32")]
	#[serde(default)]
	pub passed_user_timeout: Option<u32>,
	#[serde(rename = "trafficFilterRule")]
	pub traffic_filter_rule: Option<TrafficFilterRule>,
	#[serde(rename = "invalidateKeyUrl")]
	pub invalidate_key_url: Option<String>,
}

#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct SectionSegment {
	#[serde(deserialize_with = "deserialize_number_as_u32")]
	#[serde(default)]
	pub id: Option<u32>,
	#[serde(rename = "vwrHost")]
	pub vwr_host: Option<String>,
	#[serde(rename = "segmentName")]
	pub segment_name: Option<String>,
	#[serde(rename = "segmentKey")]
	pub segment_key: Option<String>,
	#[serde(rename = "accessYn")]
	pub access_yn: Option<bool>,
	#[serde(rename = "maxInflow")]
	#[serde(deserialize_with = "deserialize_number_as_u32")]
	#[serde(default)]
	pub max_inflow: Option<u32>,
	#[serde(rename = "startUri")]
	pub start_uri: Option<String>,
	#[serde(rename = "endUri")]
	pub end_uri: Option<String>,
	pub uris: Option<Vec<Uri>>,
	#[serde(rename = "blockingPage")]
	pub blocking_page: Option<BlockingPage>,
	pub vwr: Option<Vwr>,
	#[serde(rename = "vwrPage")]
	pub vwr_page: Option<VwrPage>,
	#[serde(rename = "passedUserTimeout")]
	#[serde(deserialize_with = "deserialize_number_as_u32")]
	#[serde(default)]
	pub passed_user_timeout: Option<u32>,
}

#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct Uri {
	#[serde(deserialize_with = "deserialize_number_as_u32")]
	#[serde(default)]
	pub id: Option<u32>,
	pub path: Option<String>,
}

#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct BlockingPage {
	#[serde(rename = "blockImagePath")]
	pub block_image_path: Option<String>,
}

#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct Vwr {
	pub block: Option<VwrBlock>,
	pub wait: Option<VwrWait>,
	pub error: Option<VwrError>,
}

#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct VwrBlock {
	#[serde(rename = "htmlUrl")]
	pub html_url: Option<String>,
	#[serde(rename = "redirectUrl")]
	pub redirect_url: Option<String>,
}

#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct VwrWait {
	#[serde(rename = "htmlUrl")]
	pub html_url: Option<String>,
	#[serde(rename = "redirectUrl")]
	pub redirect_url: Option<String>,
}

#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct VwrError {
	#[serde(rename = "htmlUrl")]
	pub html_url: Option<String>,
}

#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct VwrPage {
	#[serde(deserialize_with = "deserialize_number_as_u32")]
	#[serde(default)]
	pub id: Option<u32>,
	#[serde(rename = "logoImagePath")]
	pub logo_image_path: Option<String>,
	pub title: Option<String>,
	#[serde(rename = "followingWaitYn")]
	pub following_wait_yn: Option<bool>,
	#[serde(rename = "guideText")]
	pub guide_text: Option<String>,
	#[serde(rename = "imagePath")]
	pub image_path: Option<String>,
	#[serde(rename = "waitCancelYn")]
	pub wait_cancel_yn: Option<bool>,
	#[serde(rename = "waitTimeDisplayYn")]
	pub wait_time_display_yn: Option<bool>,
	pub name: Option<String>,
	#[serde(rename = "vwrViewLink")]
	pub vwr_view_link: Option<String>,
	#[serde(rename = "backgroundImagePath")]
	pub background_image_path: Option<String>,
}

#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct TrafficFilterRule {
	#[serde(deserialize_with = "deserialize_number_as_u32")]
	#[serde(default)]
	pub id: Option<u32>,
	#[serde(deserialize_with = "deserialize_number_as_u32")]
	#[serde(default)]
	pub priority: Option<u32>,
	#[serde(rename = "triggerRule")]
	pub trigger_rule: Option<TriggerRule>,
}

#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct TriggerRule {
	pub conditions: Vec<Condition>,
}

#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct Condition {
	pub validator: String,
	pub component: String,
	#[serde(rename = "matchOperator")]
	pub match_operator: String,
	#[serde(rename = "triggerValue")]
	pub trigger_value: String,
	#[serde(rename = "caseSensitive")]
	pub case_sensitive: bool,
	pub negate: bool,
	#[serde(rename = "logicalOperator")]
	pub logical_operator: Option<String>,
}

#[derive(Debug, Clone)]
pub struct MatchedKeys {
	pub matched_project_key: String,
	pub matched_segment_key: String,
}

#[derive(Debug, Clone)]
pub struct InvalidateKey {
	pub invalidate_project_key: String,
	pub invalidate_segment_key: String,
}
