pub mod cookie {
	use crate::constants::CookieNames;

	pub fn generate_cookie_name(project_key: &str, segment_key: &str) -> String {
		format!("{}&{}&{}", CookieNames::NF_KEY, project_key, segment_key)
	}

	pub fn generate_cookie(cookie_name: &str, is_remove: bool, cookie_value: Option<&str>) -> String {
		use crate::config::get_config;
		
		let config = get_config();
		let cookie_domain = config.as_ref().and_then(|c| if c.cookie_domain.is_empty() { None } else { Some(c.cookie_domain.as_str()) });
		
		let value = if is_remove { "" } else { cookie_value.unwrap_or("") };
		let max_age = if is_remove { "0" } else { "86400" };
		
		let mut cookie = format!("{}={}; Path=/; Max-Age={}; SameSite=Lax", cookie_name, value, max_age);
		
		if let Some(domain) = cookie_domain {
			cookie.push_str(&format!("; Domain={}", domain));
		}
		
		cookie
	}

	pub fn generate_cookie_value(key: &str, sticky: &str) -> String {
		if sticky.is_empty() {
			key.to_string()
		} else {
			format!("{}&{}", key, sticky)
		}
	}

	pub fn get_cookie_value(request: &fastly::Request, cookie_name: &str) -> Option<String> {
		let cookie_header = request.get_header("cookie")?;
		let cookie_str = cookie_header.to_str().ok()?;

		for cookie_pair in cookie_str.split(';') {
			let parts: Vec<&str> = cookie_pair.trim().splitn(2, '=').collect();
			if parts.len() == 2 && parts[0] == cookie_name {
				return Some(parts[1].to_string());
			}
		}
		None
	}
}


pub mod url_utils {
	use crate::constants::QueryNames;
	use std::collections::HashMap;
	use url::Url;

	pub fn remove_nf_queries(url: &mut Url) -> bool {
		let mut query_pairs: Vec<(String, String)> = url.query_pairs().into_owned().collect();
		let original_len = query_pairs.len();

		query_pairs.retain(|(key, _)| {
			!matches!(
				key.as_str(),
				QueryNames::NF_PJK | QueryNames::NF_SMK | QueryNames::NF_KEY | QueryNames::NF_STK
			)
		});

		if query_pairs.len() != original_len {
			let new_query = if query_pairs.is_empty() {
				None
			} else {
				Some(url::form_urlencoded::Serializer::new(String::new()).extend_pairs(query_pairs).finish())
			};
			url.set_query(new_query.as_deref());
			true
		} else {
			false
		}
	}

	pub fn get_query_params(url: &Url) -> HashMap<String, String> {
		url.query_pairs().into_owned().collect()
	}
}

pub mod rule_match {
	use crate::types::{Condition, MatchedKeys, Project};
	use regex::Regex;
	use url::Url;

	pub fn find_matched_rules(projects: &[Project], url: &str) -> Option<MatchedKeys> {
		for project in projects.iter() {
			for segment in project.segments.iter() {
				if let Some(traffic_filter) = &segment.traffic_filter_rule {
					if let Some(trigger_rule) = &traffic_filter.trigger_rule {
						if evaluate_trigger_rule(&trigger_rule.conditions, url) {
							return Some(MatchedKeys {
								matched_project_key: project.project_key.clone(),
								matched_segment_key: segment.segment_key.clone(),
							});
						}
					}
				}
			}
		}
		
		None
	}


	fn evaluate_trigger_rule(conditions: &[Condition], url: &str) -> bool {
		if conditions.is_empty() {
			return false;
		}

		let mut result = evaluate_condition(&conditions[0], url);

		for condition in conditions.iter().skip(1) {
			let condition_result = evaluate_condition(condition, url);

			if let Some(logical_op) = &condition.logical_operator {
				match logical_op.to_uppercase().as_str() {
					"AND" => {
						result = result && condition_result;
					},
					"OR" => {
						result = result || condition_result;
					},
					_ => {
						// TypeScript에서는 AND/OR가 아니면 아무것도 하지 않음
					}
				}
			}
			// logical_operator가 None이면 아무것도 하지 않음 (TypeScript와 동일)
		}

		result
	}

	fn evaluate_condition(condition: &Condition, url: &str) -> bool {
		let value = match condition.component.to_lowercase().as_str() {
			"url" => url.to_string(),
			"path" => {
				if let Ok(parsed_url) = Url::parse(url) {
					parsed_url.path().to_string()
				} else {
					// Fallback: try to extract path from URL string
					if let Some(start) = url.find("://") {
						if let Some(path_start) = url[start + 3..].find('/') {
							url[start + 3 + path_start..].split('?').next().unwrap_or("").to_string()
						} else {
							"/".to_string()
						}
					} else {
						"/".to_string()
					}
				}
			}
			"domain" => {
				if let Ok(parsed_url) = Url::parse(url) {
					parsed_url.host_str().unwrap_or("").to_string()
				} else {
					// Fallback: try to extract domain from URL string
					if let Some(start) = url.find("://") {
						let after_protocol = &url[start + 3..];
						if let Some(end) = after_protocol.find('/') {
							after_protocol[..end].split(':').next().unwrap_or("").to_string()
						} else if let Some(end) = after_protocol.find('?') {
							after_protocol[..end].split(':').next().unwrap_or("").to_string()
						} else {
							after_protocol.split(':').next().unwrap_or("").to_string()
						}
					} else {
						"".to_string()
					}
				}
			}
			"query" => {
				if let Ok(parsed_url) = Url::parse(url) {
					parsed_url.query().unwrap_or("").to_string()
				} else {
					// Fallback: try to extract query from URL string
					if let Some(query_start) = url.find('?') {
						url[query_start + 1..].split('#').next().unwrap_or("").to_string()
					} else {
						"".to_string()
					}
				}
			}
			_ => {
					url.to_string()
			}
		};

		let value = value.as_str();

		let mut matches = match condition.match_operator.as_str() {
			"Equals" => {
				if condition.case_sensitive {
					value == condition.trigger_value
				} else {
					value.to_lowercase() == condition.trigger_value.to_lowercase()
				}
			}
			"Contains" => {
				if condition.case_sensitive {
					value.contains(&condition.trigger_value)
				} else {
					value.to_lowercase().contains(&condition.trigger_value.to_lowercase())
				}
			}
			"StartsWith" => {
				if condition.case_sensitive {
					value.starts_with(&condition.trigger_value)
				} else {
					value.to_lowercase().starts_with(&condition.trigger_value.to_lowercase())
				}
			}
			"EndsWith" => {
				if condition.case_sensitive {
					value.ends_with(&condition.trigger_value)
				} else {
					value.to_lowercase().ends_with(&condition.trigger_value.to_lowercase())
				}
			}
			"Regex" => {
				let flags = if condition.case_sensitive { "" } else { "(?i)" };
				let pattern = format!("{}{}", flags, condition.trigger_value);
				if let Ok(regex) = Regex::new(&pattern) {
					regex.is_match(value)
				} else {
					false
				}
			}
			"Exists" => {
				// TypeScript: only Path and Query components are supported for Exists
				match condition.component.as_str() {
					"Path" | "Query" => !value.is_empty(),
					_ => false, // TypeScript returns false for other components
				}
			}
			_ => {
				false
			}
		};

		if condition.negate {
			matches = !matches;
		}

		matches
	}
}

pub fn is_good_bot(user_agent: &str, good_bots: &[String]) -> bool {
	let user_agent_lower = user_agent.to_lowercase();
	for bot in good_bots {
		if user_agent_lower.contains(&bot.to_lowercase()) {
			return true;
		}
	}
	false
}
